import axios from 'axios';
import { useEffect, useState } from 'react';
import { useForm } from 'react-hook-form';
import Layout from './components/Layout';
import Loading from './components/Loading';


const LinkPage = () => {

    const {
        register,
        handleSubmit,
        formState: { errors },
    } = useForm();

    const [showModal, setShowModal] = useState(false)
    const [links, setLinks] = useState([]);
    const [loading, setLoading] = useState(true);

    const onSubmitForm = async (data) => {

        try {
            const response = await axios.post(
                `${import.meta.env.VITE_API_URL}/sitelink`,
                data
            );
            // 🔥 নতুন data state-এ add করুন
            setLinks(prev => [response.data.data, ...prev]);
            alert("Sitelink Created Successfully ✅");
            setShowModal(false);

        } catch (error) {
            if (error.response) {
                console.log(error.response.data);
                alert("Error: " + error.response.data.message);
            } else {
                console.log(error.message);
            }

        }


    }

    const fetchLinks = async () => {
        try {
            const response = await axios.get(`${import.meta.env.VITE_API_URL}/site`);
            setLinks(response.data.data);
        } catch (error) {
            console.error("Error fetching links:", error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {

        fetchLinks();

    }, []);

    const handleDelete = async (id) => {
        try {
            await axios.delete(`${import.meta.env.VITE_API_URL}/sitelink/${id}`)
            // state update করে UI থেকে remove
            setLinks(prev => prev.filter(link => link.id !== id));
        } catch (error) {
            console.log(error.response?.data);
        }
    };

    if (loading) {
        return (
            <Layout>
                <Loading />
            </Layout>
        )
    }

    return (
        <Layout>
            <div className="flex items-center justify-between mb-6 border-b border-gray-200">
                <h1 className="text-3xl font-bold mb-6">Link List</h1>
                <button
                    onClick={() => setShowModal(!showModal)}
                    className="bg-indigo-500 text-white px-4 py-2 rounded-md">Add New</button>
            </div>

            {/* Link List Table */}

            <div className="bg-white rounded-2xl shadow-md border border-gray-100 overflow-hidden">

                {/* Table */}
                <div className="overflow-x-auto">
                    <table className="w-full text-sm">
                        <thead>
                            <tr className="text-gray-500 text-left bg-gray-50 uppercase border-b border-gray-200">
                                <th className="px-6 py-3">Name</th>
                                <th className="px-6 py-3">URL</th>
                                <th className="px-6 py-3 text-right">Actions</th>
                            </tr>
                        </thead>

                        <tbody className="divide-y divide-gray-100">
                            {links.map((link) => (
                                <tr key={link.id} className="hover:bg-gray-50 transition">
                                    <td className="px-6 py-4">{link.name}</td>
                                    <td className="px-6 py-4">
                                        <a
                                            href={link.site_url}
                                            target="_blank"
                                            className="text-indigo-500 hover:underline"
                                        >
                                            {link.site_url}
                                        </a>
                                    </td>
                                    <td className="px-6 py-4 text-right space-x-2">
                                        <button
                                            className="bg-red-500 hover:bg-red-600 text-white px-3 py-1.5 rounded-lg text-xs"
                                            onClick={() => handleDelete(link.id)}
                                        >
                                            Delete
                                        </button>
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* ===== Modal ===== */}
            {showModal && (
                <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/50">

                    {/* Modal Box */}
                    <div className="bg-white w-full max-w-xl rounded-2xl shadow-lg p-6 relative">

                        {/* Close Button */}
                        <button
                            onClick={() => setShowModal(false)}
                            className="absolute right-4 top-3 text-gray-400 hover:text-gray-600 text-xl"
                        >
                            ✕
                        </button>

                        <h2 className="text-xl font-semibold mb-4">Add New Link</h2>

                        <form className="space-y-4" onSubmit={handleSubmit(onSubmitForm)} errors={errors}>
                            <div>
                                <label className="block text-sm text-gray-600 mb-1">
                                    Link Name
                                </label>
                                <input
                                    {...register('name', { required: "Link name is required" })}
                                    type="text"
                                    name="name"
                                    placeholder="Enter link name"
                                    className="w-full border border-gray-300 rounded-lg px-4 py-2.5 outline-none focus:ring-2 focus:ring-blue-500"
                                />
                                <span className="text-red-500 text-sm mt-1">{errors.name?.message}</span>
                            </div>


                            <div>
                                <label className="block text-sm text-gray-600 mb-1">
                                    URL
                                </label>
                                <input
                                    {...register('site_url', { required: "URL is required" })}
                                    type="url"
                                    name="site_url"
                                    placeholder="Enter URL"
                                    className="w-full border border-gray-300 rounded-lg px-4 py-2.5 outline-none focus:ring-2 focus:ring-blue-500"
                                />
                                <span className="text-red-500 text-sm mt-1">{errors.site_url?.message}</span>
                            </div>
                            <div className="flex justify-end gap-2">
                                <button type="submit" className="bg-blue-500 text-white px-4 py-2 rounded-2xl" >Save</button>
                                <button
                                    className="bg-red-500 text-white px-4 py-2 rounded-2xl"
                                    onClick={() => setShowModal(false)}
                                >
                                    Close</button>
                            </div>
                        </form>

                    </div>
                </div>
            )}
            {/* ===== End Modal ===== */}
        </Layout>
    )
}

export default LinkPage
